import { api, APIError } from "encore.dev/api";
import { db } from "./db";

interface ImportMadrasahRequest {
  data: MadrasahImportData[];
}

interface MadrasahImportData {
  nsm: string;
  npsn: string;
  nama_madrasah: string;
  alamat?: string;
  kecamatan?: string;
  kabupaten?: string;
  provinsi?: string;
  akreditasi?: string;
  status?: string;
  nama_kepala?: string;
}

interface ImportResult {
  success: number;
  failed: number;
  errors: string[];
}

// Imports multiple madrasah records
export const importMadrasah = api<ImportMadrasahRequest, ImportResult>(
  { expose: true, method: "POST", path: "/madrasah/import" },
  async (req) => {
    const { data } = req;
    
    let success = 0;
    let failed = 0;
    const errors: string[] = [];
    
    console.log("Import madrasah request received with data:", data);
    
    for (let i = 0; i < data.length; i++) {
      const madrasah = data[i];
      try {
        console.log(`Processing madrasah ${i + 1}:`, madrasah);
        
        if (!madrasah.nsm || madrasah.nsm.trim() === "") {
          throw new Error("NSM is required");
        }
        
        if (!madrasah.npsn || madrasah.npsn.trim() === "") {
          throw new Error("NPSN is required");
        }
        
        if (!madrasah.nama_madrasah || madrasah.nama_madrasah.trim() === "") {
          throw new Error("Nama madrasah is required");
        }
        
        // Check if NSM already exists
        const existingNsm = await db.queryRow<{ id: string }>`
          SELECT id FROM madrasah WHERE nsm = ${madrasah.nsm.trim()}
        `;
        
        if (existingNsm) {
          throw new Error(`NSM ${madrasah.nsm} already exists`);
        }
        
        // Check if NPSN already exists
        const existingNpsn = await db.queryRow<{ id: string }>`
          SELECT id FROM madrasah WHERE npsn = ${madrasah.npsn.trim()}
        `;
        
        if (existingNpsn) {
          throw new Error(`NPSN ${madrasah.npsn} already exists`);
        }
        
        await db.exec`
          INSERT INTO madrasah (nsm, npsn, nama_madrasah, alamat, kecamatan, kabupaten, provinsi, akreditasi, status, nama_kepala)
          VALUES (${madrasah.nsm.trim()}, ${madrasah.npsn.trim()}, ${madrasah.nama_madrasah.trim()}, 
                  ${(madrasah.alamat || "").trim()}, ${(madrasah.kecamatan || "").trim()}, 
                  ${(madrasah.kabupaten || "").trim()}, ${(madrasah.provinsi || "").trim()}, 
                  ${(madrasah.akreditasi || "").trim()}, ${(madrasah.status || "").trim()}, 
                  ${(madrasah.nama_kepala || "").trim()})
        `;
        success++;
        console.log(`Successfully imported madrasah ${i + 1}`);
      } catch (error) {
        failed++;
        const errorMessage = error instanceof Error ? error.message : 'Unknown error';
        errors.push(`Row ${i + 1}: ${errorMessage}`);
        console.error(`Failed to import madrasah ${i + 1}:`, errorMessage);
      }
    }
    
    console.log(`Import completed. Success: ${success}, Failed: ${failed}`);
    return { success, failed, errors };
  }
);
