import { api, APIError } from "encore.dev/api";
import { SQLDatabase } from "encore.dev/storage/sqldb";

const db = SQLDatabase.named("madrasah");

interface ImportKaryawanRequest {
  madrasah_id: string;
  data: KaryawanImportData[];
}

interface KaryawanImportData {
  nip?: string;
  nama_lengkap: string;
  tempat_lahir?: string;
  tanggal_lahir?: string;
  jenis_kelamin?: string;
  agama?: string;
  alamat?: string;
  no_telepon?: string;
  email?: string;
  pendidikan_terakhir?: string;
  jabatan?: string;
  bagian?: string;
  tanggal_masuk?: string;
  status_kepegawaian?: string;
}

interface ImportResult {
  success: number;
  failed: number;
  errors: string[];
}

// Imports multiple karyawan records
export const importKaryawan = api<ImportKaryawanRequest, ImportResult>(
  { expose: true, method: "POST", path: "/karyawan/import" },
  async (req) => {
    const { madrasah_id, data } = req;
    
    let success = 0;
    let failed = 0;
    const errors: string[] = [];
    
    console.log("Import karyawan request received:", { madrasah_id, dataCount: data.length });
    
    for (let i = 0; i < data.length; i++) {
      const karyawan = data[i];
      try {
        console.log(`Processing karyawan ${i + 1}:`, karyawan);
        
        if (!karyawan.nama_lengkap || karyawan.nama_lengkap.trim() === "") {
          throw new Error("Nama lengkap is required");
        }
        
        // Parse tanggal_lahir if provided
        let tanggalLahir = null;
        if (karyawan.tanggal_lahir && karyawan.tanggal_lahir.trim() !== "") {
          const dateStr = karyawan.tanggal_lahir.trim();
          const date = new Date(dateStr);
          if (!isNaN(date.getTime())) {
            tanggalLahir = date.toISOString().split('T')[0];
          }
        }
        
        // Parse tanggal_masuk if provided
        let tanggalMasuk = null;
        if (karyawan.tanggal_masuk && karyawan.tanggal_masuk.trim() !== "") {
          const dateStr = karyawan.tanggal_masuk.trim();
          const date = new Date(dateStr);
          if (!isNaN(date.getTime())) {
            tanggalMasuk = date.toISOString().split('T')[0];
          }
        }
        
        await db.exec`
          INSERT INTO karyawan (madrasah_id, nip, nama_lengkap, tempat_lahir, tanggal_lahir, jenis_kelamin, agama, alamat, 
                               no_telepon, email, pendidikan_terakhir, jabatan, bagian, tanggal_masuk, status_kepegawaian)
          VALUES (${madrasah_id}, ${(karyawan.nip || "").trim()}, ${karyawan.nama_lengkap.trim()}, ${(karyawan.tempat_lahir || "").trim()}, 
                  ${tanggalLahir}, ${(karyawan.jenis_kelamin || "").trim()}, ${(karyawan.agama || "").trim()}, 
                  ${(karyawan.alamat || "").trim()}, ${(karyawan.no_telepon || "").trim()}, ${(karyawan.email || "").trim()}, 
                  ${(karyawan.pendidikan_terakhir || "").trim()}, ${(karyawan.jabatan || "").trim()}, 
                  ${(karyawan.bagian || "").trim()}, ${tanggalMasuk}, ${(karyawan.status_kepegawaian || "").trim()})
        `;
        success++;
        console.log(`Successfully imported karyawan ${i + 1}`);
      } catch (error) {
        failed++;
        const errorMessage = error instanceof Error ? error.message : 'Unknown error';
        errors.push(`Row ${i + 1}: ${errorMessage}`);
        console.error(`Failed to import karyawan ${i + 1}:`, errorMessage);
      }
    }
    
    console.log(`Import karyawan completed. Success: ${success}, Failed: ${failed}`);
    return { success, failed, errors };
  }
);
