import { useState } from "react";
import { useQuery } from "@tanstack/react-query";
import { useAuth } from "../contexts/AuthContext";
import { Button } from "@/components/ui/button";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Download, FileBarChart } from "lucide-react";
import backend from "~backend/client";
import * as XLSX from "xlsx";

export default function RecapDialog() {
  const { user } = useAuth();
  const [isOpen, setIsOpen] = useState(false);

  const { data: recapData, isLoading } = useQuery({
    queryKey: ["recap", user?.madrasahId],
    queryFn: () => backend.reports.getRecap({ 
      madrasah_id: user?.userType !== "superadmin" ? user?.madrasahId : undefined 
    }),
    enabled: isOpen,
  });

  const downloadExcel = () => {
    if (!recapData) return;

    const worksheetData = [
      ["Nama Madrasah", "Total Guru", "Total Siswa", "Total Karyawan", "Total Alumni"],
      ...recapData.data.map(item => [
        item.nama_madrasah || "N/A",
        item.total_guru,
        item.total_siswa,
        item.total_karyawan,
        item.total_alumni
      ]),
      [],
      ["TOTAL KESELURUHAN", 
       recapData.grand_total.total_guru,
       recapData.grand_total.total_siswa,
       recapData.grand_total.total_karyawan,
       recapData.grand_total.total_alumni
      ]
    ];

    const wb = XLSX.utils.book_new();
    const ws = XLSX.utils.aoa_to_sheet(worksheetData);
    
    // Set column widths
    ws['!cols'] = [
      { width: 30 }, // Nama Madrasah
      { width: 15 }, // Total Guru
      { width: 15 }, // Total Siswa
      { width: 15 }, // Total Karyawan
      { width: 15 }  // Total Alumni
    ];

    XLSX.utils.book_append_sheet(wb, ws, "Rekap Data");
    
    const filename = user?.userType === "superadmin" 
      ? "rekap_data_semua_madrasah.xlsx"
      : `rekap_data_${user?.madrasahName?.replace(/\s+/g, '_')}.xlsx`;
    
    XLSX.writeFile(wb, filename);
  };

  return (
    <Dialog open={isOpen} onOpenChange={setIsOpen}>
      <DialogTrigger asChild>
        <Button variant="outline" className="border-blue-600 text-blue-600 hover:bg-blue-50">
          <FileBarChart className="h-4 w-4 mr-2" />
          Rekap Data
        </Button>
      </DialogTrigger>
      <DialogContent className="max-w-4xl">
        <DialogHeader>
          <DialogTitle className="flex items-center justify-between">
            <span>Rekap Data {user?.userType === "superadmin" ? "Semua Madrasah" : user?.madrasahName}</span>
            <Button
              onClick={downloadExcel}
              disabled={!recapData || isLoading}
              className="bg-green-600 hover:bg-green-700"
              size="sm"
            >
              <Download className="h-4 w-4 mr-2" />
              Unduh Excel
            </Button>
          </DialogTitle>
        </DialogHeader>
        
        <div className="space-y-4">
          {isLoading ? (
            <div className="text-center py-8">
              <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mx-auto"></div>
              <p className="text-gray-600 mt-2">Memuat data rekap...</p>
            </div>
          ) : (
            <>
              {/* Summary Cards */}
              <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                <Card>
                  <CardHeader className="pb-2">
                    <CardTitle className="text-sm font-medium text-gray-600">Total Guru</CardTitle>
                  </CardHeader>
                  <CardContent>
                    <div className="text-2xl font-bold text-blue-600">
                      {recapData?.grand_total.total_guru.toLocaleString()}
                    </div>
                  </CardContent>
                </Card>
                <Card>
                  <CardHeader className="pb-2">
                    <CardTitle className="text-sm font-medium text-gray-600">Total Siswa</CardTitle>
                  </CardHeader>
                  <CardContent>
                    <div className="text-2xl font-bold text-green-600">
                      {recapData?.grand_total.total_siswa.toLocaleString()}
                    </div>
                  </CardContent>
                </Card>
                <Card>
                  <CardHeader className="pb-2">
                    <CardTitle className="text-sm font-medium text-gray-600">Total Karyawan</CardTitle>
                  </CardHeader>
                  <CardContent>
                    <div className="text-2xl font-bold text-orange-600">
                      {recapData?.grand_total.total_karyawan.toLocaleString()}
                    </div>
                  </CardContent>
                </Card>
                <Card>
                  <CardHeader className="pb-2">
                    <CardTitle className="text-sm font-medium text-gray-600">Total Alumni</CardTitle>
                  </CardHeader>
                  <CardContent>
                    <div className="text-2xl font-bold text-red-600">
                      {recapData?.grand_total.total_alumni.toLocaleString()}
                    </div>
                  </CardContent>
                </Card>
              </div>

              {/* Detail Table */}
              {user?.userType === "superadmin" && (
                <Card>
                  <CardHeader>
                    <CardTitle>Detail per Madrasah</CardTitle>
                  </CardHeader>
                  <CardContent>
                    <div className="overflow-x-auto">
                      <table className="w-full">
                        <thead>
                          <tr className="border-b">
                            <th className="text-left py-3 px-4 font-medium text-gray-900">Nama Madrasah</th>
                            <th className="text-left py-3 px-4 font-medium text-gray-900">Guru</th>
                            <th className="text-left py-3 px-4 font-medium text-gray-900">Siswa</th>
                            <th className="text-left py-3 px-4 font-medium text-gray-900">Karyawan</th>
                            <th className="text-left py-3 px-4 font-medium text-gray-900">Alumni</th>
                          </tr>
                        </thead>
                        <tbody>
                          {recapData?.data.map((item) => (
                            <tr key={item.madrasah_id} className="border-b hover:bg-gray-50">
                              <td className="py-3 px-4 text-sm font-medium">{item.nama_madrasah}</td>
                              <td className="py-3 px-4 text-sm text-blue-600">{item.total_guru.toLocaleString()}</td>
                              <td className="py-3 px-4 text-sm text-green-600">{item.total_siswa.toLocaleString()}</td>
                              <td className="py-3 px-4 text-sm text-orange-600">{item.total_karyawan.toLocaleString()}</td>
                              <td className="py-3 px-4 text-sm text-red-600">{item.total_alumni.toLocaleString()}</td>
                            </tr>
                          ))}
                        </tbody>
                      </table>
                    </div>
                  </CardContent>
                </Card>
              )}
            </>
          )}
        </div>
      </DialogContent>
    </Dialog>
  );
}
