import { api, APIError } from "encore.dev/api";
import { SQLDatabase } from "encore.dev/storage/sqldb";

const db = SQLDatabase.named("madrasah");

interface ImportAlumniRequest {
  madrasah_id: string;
  data: AlumniImportData[];
}

interface AlumniImportData {
  nisn?: string;
  nis?: string;
  nama_lengkap: string;
  tempat_lahir?: string;
  tanggal_lahir?: string;
  jenis_kelamin?: string;
  agama?: string;
  alamat?: string;
  no_telepon?: string;
  tahun_lulus: number;
  melanjutkan_ke?: string;
  pekerjaan?: string;
  alamat_sekarang?: string;
}

interface ImportResult {
  success: number;
  failed: number;
  errors: string[];
}

// Imports multiple alumni records
export const importAlumni = api<ImportAlumniRequest, ImportResult>(
  { expose: true, method: "POST", path: "/alumni/import" },
  async (req) => {
    const { madrasah_id, data } = req;
    
    let success = 0;
    let failed = 0;
    const errors: string[] = [];
    
    console.log("Import alumni request received:", { madrasah_id, dataCount: data.length });
    
    for (let i = 0; i < data.length; i++) {
      const alumni = data[i];
      try {
        console.log(`Processing alumni ${i + 1}:`, alumni);
        
        if (!alumni.nama_lengkap || alumni.nama_lengkap.trim() === "") {
          throw new Error("Nama lengkap is required");
        }
        
        // Parse tahun_lulus
        let tahunLulus = null;
        if (alumni.tahun_lulus) {
          const year = parseInt(String(alumni.tahun_lulus));
          if (!isNaN(year) && year > 1900 && year < 2100) {
            tahunLulus = year;
          } else {
            throw new Error("Tahun lulus harus berupa angka yang valid");
          }
        } else {
          throw new Error("Tahun lulus is required");
        }
        
        // Parse tanggal_lahir if provided
        let tanggalLahir = null;
        if (alumni.tanggal_lahir && alumni.tanggal_lahir.trim() !== "") {
          const dateStr = alumni.tanggal_lahir.trim();
          const date = new Date(dateStr);
          if (!isNaN(date.getTime())) {
            tanggalLahir = date.toISOString().split('T')[0];
          }
        }
        
        await db.exec`
          INSERT INTO alumni (madrasah_id, nisn, nis, nama_lengkap, tempat_lahir, tanggal_lahir, jenis_kelamin, agama, alamat, 
                             no_telepon, tahun_lulus, melanjutkan_ke, pekerjaan, alamat_sekarang)
          VALUES (${madrasah_id}, ${(alumni.nisn || "").trim()}, ${(alumni.nis || "").trim()}, ${alumni.nama_lengkap.trim()}, 
                  ${(alumni.tempat_lahir || "").trim()}, ${tanggalLahir}, ${(alumni.jenis_kelamin || "").trim()}, 
                  ${(alumni.agama || "").trim()}, ${(alumni.alamat || "").trim()}, ${(alumni.no_telepon || "").trim()}, 
                  ${tahunLulus}, ${(alumni.melanjutkan_ke || "").trim()}, ${(alumni.pekerjaan || "").trim()}, 
                  ${(alumni.alamat_sekarang || "").trim()})
        `;
        success++;
        console.log(`Successfully imported alumni ${i + 1}`);
      } catch (error) {
        failed++;
        const errorMessage = error instanceof Error ? error.message : 'Unknown error';
        errors.push(`Row ${i + 1}: ${errorMessage}`);
        console.error(`Failed to import alumni ${i + 1}:`, errorMessage);
      }
    }
    
    console.log(`Import alumni completed. Success: ${success}, Failed: ${failed}`);
    return { success, failed, errors };
  }
);
